/**
 * editor.js
 * Emmet for gPad
 */

var editorProxy = emmet.exec(function(require, _){
	var context = null;
	
	function PointToSerial( x, y )
	{
		nIndex = 0;
		for( i = 1; i < y; i++ ){
			s = document.GetLine( i, eGetLineWithNewLines );
			nIndex += s.length;
		}
		nIndex += (x - 1);  // x is one-base index.
		return nIndex;
	}
	
	function SerialToPoint( nIndex )
	{
		nOrgIndex = nIndex;
		x = 0;
		y = 0;
		yLines = document.GetLines();
		s = "";
		for( i = 1; i <= yLines; i++ ){
			s = document.GetLine( i, eGetLineWithNewLines );
			if( nIndex < s.length ){
				x = nIndex + 1;
				y = i;
				return {
					x: x,
					y: y
				}
			}
			nIndex -= s.length;
		}
		
		x = s.length + 1;
		y = yLines;
		return {
			x: x,
			y: y
		}
	}
	
	/**
	 * Returns whitrespace padding of string
	 * @param {String} str String line
	 * @return {String}
	 */
	function getStringPadding(str) {
		return (str.match(/^(\s+)/) || [''])[0];
	}

	/**
	 * Handle tab-stops (like $1 or ${1:label}) inside text: find first tab-stop,
	 * marks it as selection, remove the rest. If tab-stop wasn't found, search
	 * for caret placeholder and use it as selection
	 * @param {String} text
	 * @return {Array} Array with new text and selection indexes (['...', -1,-1] 
	 * if there's no selection)
	 */
	function handleTabStops(text) {
		var utils = require('utils');

		var selection_len = 0,
			caret_placeholder = utils.getCaretPlaceholder(),
			caret_pos = text.indexOf(caret_placeholder),
			placeholders = {};
			
		// find caret position
		if (caret_pos != -1) {
			text = text.split(caret_placeholder).join('');
		} else {
			caret_pos = text.length;
		}
		
		text = processTextBeforePaste(text, 
			function(ch){ return ch; }, 
			function(i, num, val) {
				if (val) placeholders[num] = val;
				
				if (i < caret_pos) {
					caret_pos = i;
					if (val)
						selection_len = val.length;
				}
					
				return placeholders[num] || '';
			});
		
		return [text, caret_pos, caret_pos + selection_len];
	}

	/**
	 * Process text that should be pasted into editor: clear escaped text and
	 * handle tabstops
	 * @param {String} text
	 * @param {Function} escape_fn Handle escaped character. Must return
	 * replaced value
	 * @param {Function} tabstop_fn Callback function that will be called on every
	 * tabstob occurance, passing <b>index</b>, <code>number</code> and 
	 * <b>value</b> (if exists) arguments. This function must return 
	 * replacement value
	 * @return {String} 
	 */
	function processTextBeforePaste(text, escape_fn, tabstop_fn) {
		var i = 0, il = text.length, start_ix, _i,
			str_builder = [];
			
		var nextWhile = function(ix, fn) {
			while (ix < il) if (!fn(text.charAt(ix++))) break;
			return ix - 1;
		};
		
		while (i < il) {
			var ch = text.charAt(i);
			if (ch == '\\' && i + 1 < il) {
				// handle escaped character
				str_builder.push(escape_fn(text.charAt(i + 1)));
				i += 2;
				continue;
			} else if (ch == '$') {
				// looks like a tabstop
				var next_ch = text.charAt(i + 1) || '';
				_i = i;
				if (this.isNumeric(next_ch)) {
					// $N placeholder
					start_ix = i + 1;
					i = nextWhile(start_ix, this.isNumeric);
					if (start_ix < i) {
						str_builder.push(tabstop_fn(_i, text.substring(start_ix, i)));
						continue;
					}
				} else if (next_ch == '{') {
					// ${N:value} or ${N} placeholder
					var brace_count = 1;
					start_ix = i + 2;
					i = nextWhile(start_ix, this.isNumeric);
					
					if (i > start_ix) {
						if (text.charAt(i) == '}') {
							str_builder.push(tabstop_fn(_i, text.substring(start_ix, i)));
							i++; // handle closing brace
							continue;
						} else if (text.charAt(i) == ':') {
							var val_start = i + 2;
							i = nextWhile(val_start, function(c) {
								if (c == '{') brace_count++;
								else if (c == '}') brace_count--;
								return !!brace_count;
							});
							str_builder.push(tabstop_fn(_i, text.substring(start_ix, val_start - 2), text.substring(val_start - 1, i)));
							i++; // handle closing brace
							continue;
						}
					}
				}
				i = _i;
			}
			
			// push current character to stack
			str_builder.push(ch);
			i++;
		}
		
		return str_builder.join('');
	}
	return {
		/**
		 * Setup underlying editor context. You should call this method 
		 * <code>before</code> using any Emmet action.
		 * @param {Object} context
		 */
		setContext: function(ctx) {
			context = ctx;
		},
		
		/**
		 * Returns character indexes of selected text: object with <code>start</code>
		 * and <code>end</code> properties. If there's no selection, should return 
		 * object with <code>start</code> and <code>end</code> properties referring
		 * to current caret position
		 * @return {Object}
		 * @example
		 * var selection = editor.getSelectionRange();
		 * alert(selection.start + ', ' + selection.end); 
		 */
		getSelectionRange: function() {
			x = document.selection.GetTopPointX();
			y = document.selection.GetTopPointY();
			start = PointToSerial( x, y );
			x = document.selection.GetBottomPointX();
			y = document.selection.GetBottomPointY();
			end = PointToSerial( x, y );
			
			return {
				start: start, 
				end: end
			};
		},
		
		/**
		 * Creates selection from <code>start</code> to <code>end</code> character
		 * indexes. If <code>end</code> is ommited, this method should place caret 
		 * and <code>start</code> index
		 * @param {Number} start
		 * @param {Number} [end]
		 * @example
		 * editor.createSelection(10, 40);
		 * 
		 * //move caret to 15th character
		 * editor.createSelection(15);
		 */
		createSelection: function(start, end) {
			var pt = SerialToPoint( start );
			document.selection.SetActivePoint( ePosLogical, pt.x, pt.y );
			pt = SerialToPoint( end );
			document.selection.SetActivePoint( ePosLogical, pt.x, pt.y, true );
		},
		
		/**
		 * Returns current line's start and end indexes as object with <code>start</code>
		 * and <code>end</code> properties
		 * @return {Object}
		 * @example
		 * var range = editor.getCurrentLineRange();
		 * alert(range.start + ', ' + range.end);
		 */
		getCurrentLineRange: function() {
			return require('utils').findNewlineBounds(this.getContent(), this.getCaretPos());
		},
		
		/**
		 * Returns current caret position
		 * @return {Number}
		 */
		getCaretPos: function(){
			x = document.selection.GetActivePointX( ePosLogical );
			y = document.selection.GetActivePointY( ePosLogical );
			return PointToSerial( x, y );
		},
		
		/**
		 * Set new caret position
		 * @param {Number} pos Caret position
		 */
		setCaretPos: function(pos) {
			this.createSelection(pos, pos);
		},
		
		/**
		 * Returns content of current line
		 * @return {String}
		 */
		getCurrentLine: function() {
			var range = this.getCurrentLineRange();
			return range.start < range.end ? this.getContent().substring(range.start, range.end) : '';
		},
		
		/**
		 * Replace editor's content or it's part (from <code>start</code> to 
		 * <code>end</code> index). If <code>value</code> contains 
		 * <code>caret_placeholder</code>, the editor will put caret into 
		 * this position. If you skip <code>start</code> and <code>end</code>
		 * arguments, the whole target's content will be replaced with 
		 * <code>value</code>. 
		 * 
		 * If you pass <code>start</code> argument only,
		 * the <code>value</code> will be placed at <code>start</code> string 
		 * index of current content. 
		 * 
		 * If you pass <code>start</code> and <code>end</code> arguments,
		 * the corresponding substring of current target's content will be 
		 * replaced with <code>value</code>. 
		 * @param {String} value Content you want to paste
		 * @param {Number} start Start index of editor's content
		 * @param {Number} end End index of editor's content
		 */
		replaceContent: function(value, start, end, noIndent) {
			var utils = require('utils');

			var content = this.getContent(),
				caret_pos = this.getCaretPos(),
				caret_placeholder = utils.getCaretPlaceholder(),
				has_start = typeof(start) !== 'undefined',
				has_end = typeof(end) !== 'undefined';
				
			// indent new value
			if (!noIndent) {
				//value = utils.padString(value, getStringPadding(this.getCurrentLine()));
				value = utils.padString(value, utils.getLinePaddingFromPosition(content, start));
			}
			
			// find new caret position
			var tabstop_res = handleTabStops(value);
			value = tabstop_res[0];
			
			start = start || 0;
			if (tabstop_res[1] !== -1) {
				tabstop_res[1] += start;
				tabstop_res[2] += start;
			} else {
				tabstop_res[1] = tabstop_res[2] = value.length + start;
			}
			
			try {
				if (has_start && has_end) {
					content = content.substring(0, start) + value + content.substring(end);
				} else if (has_start) {
					content = content.substring(0, start) + value + content.substring(start);
				}
				
				this.createSelection(start, end);
				document.selection.Text = value;
				this.createSelection(tabstop_res[1], tabstop_res[2]);
			} catch(e){}
		},
		
		/**
		 * Returns editor's content
		 * @return {String}
		 */
		getContent: function(){
			return document.Text || '';
		},
		
		/**
		 * Returns current editor's syntax mode
		 * @return {String}
		 */
		getSyntax: function() {
			var m = /\.(\w+)$/.exec(this.getFilePath());
			return require('actionUtils').detectSyntax(this, m ? m[1].toLowerCase() : null);
		},
		
		/**
		 * Returns current output profile name (@see emmet#setupProfile)
		 * @return {String}
		 */
		getProfileName: function() {
			return require('actionUtils').detectProfile(this);
		},
		
		/**
		 * Ask user to enter something
		 * @param {String} title Dialog title
		 * @return {String} Entered data
		 * @since 0.65
		 */
		prompt: function(title) {
			return inputText(title);
		},
		
		/**
		 * Returns current selection
		 * @return {String}
		 * @since 0.65
		 */
		getSelection: function() {
			var sel = getSelectionRange();
			if (sel) {
				try {
					return getContent().substring(sel.start, sel.end);
				} catch(e) {}
			}
			
			return '';
		},
		
		/**
		 * Returns current editor's file path
		 * @return {String}
		 * @since 0.65 
		 */
		getFilePath: function() {
			return document.FullName;
		}
	};
});
